<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use app\models\Booking;
use app\models\User;

class AdminController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'matchCallback' => function ($rule, $action) {
                            return !Yii::$app->user->isGuest && 
                                   Yii::$app->user->identity->username === 'hotel';
                        },
                    ],
                ],
                'denyCallback' => function () {
                    Yii::$app->session->setFlash('error', 'Доступ запрещен');
                    return Yii::$app->user->isGuest ? 
                        $this->redirect(['/site/login']) : 
                        $this->redirect(['/site/index']);
                },
            ],
        ];
    }

    /**
     * Главная страница панели администратора
     */
    public function actionIndex()
    {
        $bookingsCount = Booking::find()->count();
        $usersCount = User::find()->count();
        $newBookingsCount = Booking::find()->where(['status' => Booking::STATUS_NEW])->count();
        
        return $this->render('index', [
            'bookingsCount' => $bookingsCount,
            'usersCount' => $usersCount,
            'newBookingsCount' => $newBookingsCount,
        ]);
    }

    /**
     * Управление бронированиями
     */
    public function actionBookings()
    {
        $query = Booking::find()
            ->with(['user', 'roomType'])
            ->orderBy(['created_at' => SORT_DESC]);

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('bookings', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Изменение статуса бронирования
     */
    public function actionUpdateStatus($id, $status)
    {
        $booking = Booking::findOne($id);
        
        if (!$booking) {
            Yii::$app->session->setFlash('error', 'Бронирование не найдено');
            return $this->redirect(['bookings']);
        }

        $allowedStatuses = [
            Booking::STATUS_NEW,
            Booking::STATUS_PROCESSING,
            Booking::STATUS_CONFIRMED,
            Booking::STATUS_CANCELLED,
            Booking::STATUS_COMPLETED
        ];

        if (in_array($status, $allowedStatuses)) {
            $booking->status = $status;
            if ($booking->save()) {
                Yii::$app->session->setFlash('success', 'Статус обновлен успешно');
            } else {
                Yii::$app->session->setFlash('error', 'Ошибка при обновлении статуса');
            }
        } else {
            Yii::$app->session->setFlash('error', 'Недопустимый статус');
        }

        return $this->redirect(['bookings']);
    }

    /**
     * Просмотр подробной информации о бронировании
     */
    public function actionViewBooking($id)
    {
        $booking = Booking::find()
            ->with(['user', 'roomType', 'services', 'review'])
            ->where(['id' => $id])
            ->one();

        if (!$booking) {
            throw new \yii\web\NotFoundHttpException('Бронирование не найдено');
        }

        return $this->render('view-booking', [
            'booking' => $booking,
        ]);
    }

    /**
     * Управление пользователями
     */
    public function actionUsers()
    {
        $query = User::find()
            ->orderBy(['created_at' => SORT_DESC]);

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 20,
            ],
        ]);

        return $this->render('users', [
            'dataProvider' => $dataProvider,
        ]);
    }
}