<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * Class Booking
 * @package app\models
 */
class Booking extends ActiveRecord
{
    const STATUS_NEW = 'new';
    const STATUS_PROCESSING = 'processing';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_CANCELLED = 'cancelled';
    const STATUS_COMPLETED = 'completed';

    const PAYMENT_CASH = 'cash';
    const PAYMENT_CARD = 'card';

    public $additional_services;
    public $service_description;

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%booking}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['room_type_id', 'check_in_date', 'check_out_date', 'adults_count', 'payment_method'], 'required'],
            [['room_type_id'], 'integer'],
            ['adults_count', 'integer', 'min' => 1, 'max' => 10],
            ['children_count', 'integer', 'min' => 0, 'max' => 10],
            [['check_in_date', 'check_out_date'], 'date', 'format' => 'php:Y-m-d'],
            ['check_out_date', 'compare', 'compareAttribute' => 'check_in_date', 'operator' => '>', 
             'message' => 'Дата выезда должна быть позже даты заезда'],
            ['payment_method', 'in', 'range' => [self::PAYMENT_CASH, self::PAYMENT_CARD]],
            [['special_requests', 'service_description'], 'string'],
            ['additional_services', 'boolean'],
            ['children_count', 'default', 'value' => 0],
            [['created_at', 'updated_at'], 'safe'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'room_type_id' => 'Тип номера',
            'check_in_date' => 'Дата заезда',
            'check_out_date' => 'Дата выезда',
            'adults_count' => 'Взрослых',
            'children_count' => 'Детей',
            'total_price' => 'Общая стоимость',
            'payment_method' => 'Способ оплаты',
            'status' => 'Статус',
            'special_requests' => 'Особые пожелания',
            'additional_services' => 'Дополнительные услуги',
            'service_description' => 'Описание услуги',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeValidate()
    {
        if (!parent::beforeValidate()) {
            return false;
        }

        // Преобразуем children_count в 0 если пустое значение
        if ($this->children_count === '' || $this->children_count === null) {
            $this->children_count = 0;
        }

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->user_id = Yii::$app->user->id;
                $this->created_at = time();
                $this->status = self::STATUS_NEW;
            }
            $this->updated_at = time();

            // Валидация максимального количества гостей
            $roomType = RoomType::findOne($this->room_type_id);
            if ($roomType) {
                $totalGuests = $this->adults_count + $this->children_count;
                if ($totalGuests > $roomType->max_guests) {
                    $this->addError('adults_count', 
                        "Максимальное количество гостей для этого номера: {$roomType->max_guests}");
                    return false;
                }
            }

            // Расчет общей стоимости
            if ($this->check_in_date && $this->check_out_date && $this->room_type_id) {
                $checkIn = new \DateTime($this->check_in_date);
                $checkOut = new \DateTime($this->check_out_date);
                $nights = $checkOut->diff($checkIn)->days;
                
                if ($nights > 0) {
                    if ($roomType) {
                        $this->total_price = $nights * $roomType->price_per_night;
                    }
                } else {
                    $this->addError('check_out_date', 'Дата выезда должна быть минимум на 1 день позже даты заезда');
                    return false;
                }
            }

            return true;
        }
        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        
        // Сохранение дополнительных услуг
        if ($this->additional_services && !empty($this->service_description)) {
            $service = new BookingService();
            $service->booking_id = $this->id;
            $service->service_name = 'Дополнительная услуга';
            $service->service_description = $this->service_description;
            $service->created_at = time();
            
            if (!$service->save()) {
                Yii::error('Ошибка сохранения услуги: ' . print_r($service->errors, true));
            }
        }
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getRoomType()
    {
        return $this->hasOne(RoomType::class, ['id' => 'room_type_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getServices()
    {
        return $this->hasMany(BookingService::class, ['booking_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getReview()
    {
        return $this->hasOne(Review::class, ['booking_id' => 'id']);
    }

    /**
     * Получить статус в читаемом формате
     * @return string
     */
    public function getStatusLabel()
    {
        $statuses = [
            self::STATUS_NEW => 'Новая',
            self::STATUS_PROCESSING => 'В процессе',
            self::STATUS_CONFIRMED => 'Подтверждена',
            self::STATUS_CANCELLED => 'Отменена',
            self::STATUS_COMPLETED => 'Завершена',
        ];
        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Получить способ оплаты в читаемом формате
     * @return string
     */
    public function getPaymentMethodLabel()
    {
        $methods = [
            self::PAYMENT_CASH => 'Наличные',
            self::PAYMENT_CARD => 'Банковская карта',
        ];
        return $methods[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * Получить общее количество гостей
     * @return int
     */
    public function getTotalGuests()
    {
        return $this->adults_count + $this->children_count;
    }

    /**
     * Получить количество ночей
     * @return int
     */
    public function getNightsCount()
    {
        if (!$this->check_in_date || !$this->check_out_date) {
            return 0;
        }
        
        $checkIn = new \DateTime($this->check_in_date);
        $checkOut = new \DateTime($this->check_out_date);
        return $checkOut->diff($checkIn)->days;
    }

    /**
     * Получить форматированную общую стоимость
     * @return string
     */
    public function getFormattedTotalPrice()
    {
        return number_format($this->total_price, 0, ',', ' ') . ' ₽';
    }

    /**
     * Проверка, можно ли оставить отзыв
     * @return bool
     */
    public function canReview()
    {
        return $this->status === self::STATUS_COMPLETED && !$this->review;
    }

    /**
     * Проверка возможности бронирования
     * @return bool
     */
    public function validateMaxGuests()
    {
        if (!$this->roomType) {
            return false;
        }
        
        return $this->getTotalGuests() <= $this->roomType->max_guests;
    }
}