<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

class Room extends ActiveRecord
{
    const STATUS_AVAILABLE = 'available';
    const STATUS_OCCUPIED = 'occupied';
    const STATUS_MAINTENANCE = 'maintenance';

    public static function tableName()
    {
        return '{{%room}}';
    }

    public function rules()
    {
        return [
            [['room_type_id', 'room_number', 'floor'], 'required'],
            [['room_type_id', 'floor'], 'integer'],
            [['room_number'], 'string', 'max' => 10],
            [['room_number'], 'unique'],
            ['status', 'in', 'range' => [self::STATUS_AVAILABLE, self::STATUS_OCCUPIED, self::STATUS_MAINTENANCE]],
            ['status', 'default', 'value' => self::STATUS_AVAILABLE],
        ];
    }

    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'room_type_id' => 'Тип номера',
            'room_number' => 'Номер комнаты',
            'floor' => 'Этаж',
            'status' => 'Статус',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->created_at = time();
            }
            $this->updated_at = time();
            return true;
        }
        return false;
    }

    // Relations
    public function getRoomType()
    {
        return $this->hasOne(RoomType::class, ['id' => 'room_type_id']);
    }

    public function getStatusLabel()
    {
        $statuses = [
            self::STATUS_AVAILABLE => 'Доступен',
            self::STATUS_OCCUPIED => 'Занят',
            self::STATUS_MAINTENANCE => 'На обслуживании',
        ];
        return $statuses[$this->status] ?? $this->status;
    }

    public function isAvailableForDates($checkIn, $checkOut)
    {
        if ($this->status !== self::STATUS_AVAILABLE) {
            return false;
        }

        // Проверка на пересечение с существующими бронированиями
        $existingBookings = Booking::find()
            ->joinWith('roomType.rooms')
            ->where(['room.id' => $this->id])
            ->andWhere(['or',
                ['and',
                    ['<=', 'check_in_date', $checkIn],
                    ['>=', 'check_out_date', $checkIn]
                ],
                ['and',
                    ['<=', 'check_in_date', $checkOut],
                    ['>=', 'check_out_date', $checkOut]
                ],
                ['and',
                    ['>=', 'check_in_date', $checkIn],
                    ['<=', 'check_out_date', $checkOut]
                ]
            ])
            ->andWhere(['not in', 'status', [Booking::STATUS_CANCELLED]])
            ->exists();

        return !$existingBookings;
    }
}