<?php

/* @var $this yii\web\View */
/* @var $model app\models\Booking */
/* @var $roomTypes app\models\RoomType[] */
/* @var $minDate string */
/* @var $maxDate string */

use yii\helpers\Html;
use yii\helpers\Url;

$this->title = 'Бронирование номера';
?>
<div class="booking-create">
    <h1 class="page-title"><?= Html::encode($this->title) ?></h1>

    <div class="booking-form-container">
        <form id="booking-form" class="booking-form" method="post" action="<?= Url::to(['/booking/create']) ?>">
            <input type="hidden" name="<?= Yii::$app->request->csrfParam ?>" value="<?= Yii::$app->request->csrfToken ?>">
            
            <?php if ($model->hasErrors()): ?>
                <div class="alert alert-error">
                    <h3>Обнаружены ошибки:</h3>
                    <ul>
                        <?php foreach ($model->getErrors() as $attribute => $errors): ?>
                            <?php foreach ($errors as $error): ?>
                                <li><?= Html::encode($model->getAttributeLabel($attribute) . ': ' . $error) ?></li>
                            <?php endforeach; ?>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <div class="form-section">
                <h2 class="section-title">Выбор номера</h2>
                
                <div class="form-group">
                    <label for="booking-room_type_id" class="form-label">Тип номера *</label>
                    <select id="booking-room_type_id" class="form-select" name="Booking[room_type_id]" required>
                        <option value="">Выберите тип номера</option>
                        <?php foreach ($roomTypes as $roomType): ?>
                            <option value="<?= $roomType->id ?>" 
                                    data-price="<?= $roomType->price_per_night ?>"
                                    data-max-guests="<?= $roomType->max_guests ?>"
                                    data-description="<?= Html::encode($roomType->description) ?>"
                                    data-bed-type="<?= Html::encode($roomType->bed_type) ?>"
                                    data-room-size="<?= Html::encode($roomType->room_size) ?>"
                                    data-amenities="<?= Html::encode($roomType->amenities) ?>"
                                    <?= $model->room_type_id == $roomType->id ? 'selected' : '' ?>>
                                <?= Html::encode($roomType->name) ?> - <?= $roomType->getFormattedPrice() ?> / ночь
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <?= Html::error($model, 'room_type_id', ['class' => 'error-text']) ?>
                    <div class="room-info" id="room-info" style="display: none;">
                        <div class="room-details">
                            <p id="room-description"></p>
                            <p><strong>Максимум гостей:</strong> <span id="room-max-guests"></span></p>
                            <p><strong>Тип кровати:</strong> <span id="room-bed-type"></span></p>
                            <p><strong>Размер номера:</strong> <span id="room-size"></span></p>
                            <p><strong>Удобства:</strong> <span id="room-amenities"></span></p>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="form-section">
                <h2 class="section-title">Даты проживания</h2>
                
                <div class="date-fields">
                    <div class="form-group">
                        <label for="booking-check_in_date" class="form-label">Дата заезда *</label>
                        <input type="date" id="booking-check_in_date" class="form-input" 
                               name="Booking[check_in_date]" required
                               min="<?= $minDate ?>" max="<?= $maxDate ?>"
                               value="<?= Html::encode($model->check_in_date) ?>">
                        <?= Html::error($model, 'check_in_date', ['class' => 'error-text']) ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="booking-check_out_date" class="form-label">Дата выезда *</label>
                        <input type="date" id="booking-check_out_date" class="form-input" 
                               name="Booking[check_out_date]" required
                               min="<?= $minDate ?>" max="<?= $maxDate ?>"
                               value="<?= Html::encode($model->check_out_date) ?>">
                        <?= Html::error($model, 'check_out_date', ['class' => 'error-text']) ?>
                    </div>
                </div>
                
                <div id="booking-summary" class="booking-summary" style="display: none;">
                    <div class="summary-item">
                        <span>Количество ночей:</span>
                        <strong id="nights-count">0</strong>
                    </div>
                    <div class="summary-item">
                        <span>Цена за ночь:</span>
                        <strong id="price-per-night">0 ₽</strong>
                    </div>
                    <div class="summary-item total">
                        <span>Общая стоимость:</span>
                        <strong id="total-price">0 ₽</strong>
                    </div>
                </div>
            </div>
            
            <div class="form-section">
                <h2 class="section-title">Информация о гостях</h2>
                
                <div class="guest-fields">
                    <div class="form-group">
                        <label for="booking-adults_count" class="form-label">Взрослых *</label>
                        <input type="number" id="booking-adults_count" class="form-input" 
                               name="Booking[adults_count]" min="1" max="10" value="<?= Html::encode($model->adults_count ?: 1) ?>" required>
                        <?= Html::error($model, 'adults_count', ['class' => 'error-text']) ?>
                    </div>
                    
                    <div class="form-group">
                        <label for="booking-children_count" class="form-label">Детей</label>
                        <input type="number" id="booking-children_count" class="form-input" 
                               name="Booking[children_count]" min="0" max="10" value="<?= Html::encode($model->children_count ?: 0) ?>">
                        <?= Html::error($model, 'children_count', ['class' => 'error-text']) ?>
                    </div>
                </div>
                
                <div id="guest-warning" class="alert alert-warning" style="display: none;">
                    <p>Количество гостей превышает максимально допустимое для выбранного номера.</p>
                </div>
            </div>
            
            <div class="form-section">
                <h2 class="section-title">Дополнительные услуги</h2>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" id="booking-additional_services" 
                               name="Booking[additional_services]" value="1"
                               <?= $model->additional_services ? 'checked' : '' ?>>
                        <span>Требуются дополнительные услуги</span>
                    </label>
                </div>
                
                <div id="service-description-field" class="form-group" style="display: <?= $model->additional_services ? 'block' : 'none' ?>;">
                    <label for="booking-service_description" class="form-label">Опишите, какие услуги вам необходимы:</label>
                    <textarea id="booking-service_description" class="form-input" 
                              name="Booking[service_description]" rows="3"><?= Html::encode($model->service_description) ?></textarea>
                    <?= Html::error($model, 'service_description', ['class' => 'error-text']) ?>
                </div>
            </div>
            
            <div class="form-section">
                <h2 class="section-title">Способ оплаты</h2>
                
                <div class="payment-methods">
                    <label class="radio-label">
                        <input type="radio" name="Booking[payment_method]" value="cash" 
                               <?= ($model->payment_method ?: 'cash') === 'cash' ? 'checked' : '' ?>>
                        <span>Наличные (при заселении)</span>
                    </label>
                    
                    <label class="radio-label">
                        <input type="radio" name="Booking[payment_method]" value="card"
                               <?= $model->payment_method === 'card' ? 'checked' : '' ?>>
                        <span>Банковская карта</span>
                    </label>
                </div>
                <?= Html::error($model, 'payment_method', ['class' => 'error-text']) ?>
            </div>
            
            <div class="form-section">
                <h2 class="section-title">Особые пожелания</h2>
                
                <div class="form-group">
                    <label for="booking-special_requests" class="form-label">Если у вас есть особые пожелания, укажите их:</label>
                    <textarea id="booking-special_requests" class="form-input" 
                              name="Booking[special_requests]" rows="3"><?= Html::encode($model->special_requests) ?></textarea>
                    <?= Html::error($model, 'special_requests', ['class' => 'error-text']) ?>
                </div>
            </div>
            
            <div class="form-actions">
                <button type="submit" class="btn btn-primary btn-large">Забронировать</button>
                <a href="<?= Url::to(['/booking/index']) ?>" class="btn btn-secondary">Отмена</a>
            </div>
        </form>
    </div>
</div>

<style>
/* ... существующие стили ... */

.alert-error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
    padding: 1rem;
    border-radius: 4px;
    margin-bottom: 2rem;
}

.alert-error h3 {
    margin-top: 0;
    margin-bottom: 0.5rem;
    font-size: 1.1rem;
}

.alert-error ul {
    margin: 0;
    padding-left: 1.5rem;
}

.alert-error li {
    margin-bottom: 0.25rem;
}

.error-text {
    color: #e74c3c;
    font-size: 0.85rem;
    margin-top: 0.25rem;
    display: block;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('booking-form');
    const roomSelect = document.getElementById('booking-room_type_id');
    const roomInfo = document.getElementById('room-info');
    const checkInInput = document.getElementById('booking-check_in_date');
    const checkOutInput = document.getElementById('booking-check_out_date');
    const adultsInput = document.getElementById('booking-adults_count');
    const childrenInput = document.getElementById('booking-children_count');
    const additionalServicesCheckbox = document.getElementById('booking-additional_services');
    const serviceDescriptionField = document.getElementById('service-description-field');
    const bookingSummary = document.getElementById('booking-summary');
    const guestWarning = document.getElementById('guest-warning');
    
    // Собираем данные о номерах
    const roomData = {};
    <?php foreach ($roomTypes as $roomType): ?>
        roomData[<?= $roomType->id ?>] = {
            price: <?= $roomType->price_per_night ?>,
            maxGuests: <?= $roomType->max_guests ?>,
            description: "<?= addslashes($roomType->description) ?>",
            bedType: "<?= addslashes($roomType->bed_type) ?>",
            roomSize: "<?= addslashes($roomType->room_size) ?>",
            amenities: "<?= addslashes($roomType->amenities) ?>"
        };
    <?php endforeach; ?>
    
    // Обновление информации о номере при загрузке страницы
    updateRoomInfo();
    
    // Обновление информации о номере
    roomSelect.addEventListener('change', updateRoomInfo);
    
    // Обновление расчета стоимости при изменении дат
    checkInInput.addEventListener('change', updateBookingSummary);
    checkOutInput.addEventListener('change', updateBookingSummary);
    
    // Проверка количества гостей
    adultsInput.addEventListener('input', checkGuestsCount);
    childrenInput.addEventListener('input', checkGuestsCount);
    
    // Показ/скрытие поля для описания услуг
    additionalServicesCheckbox.addEventListener('change', function() {
        serviceDescriptionField.style.display = this.checked ? 'block' : 'none';
        if (this.checked) {
            document.getElementById('booking-service_description').focus();
        }
    });
    
    // Валидация формы
    form.addEventListener('submit', function(e) {
        // Базовые проверки перед отправкой
        const roomId = roomSelect.value;
        const checkIn = checkInInput.value;
        const checkOut = checkOutInput.value;
        const adults = parseInt(adultsInput.value);
        
        if (!roomId || !checkIn || !checkOut || !adults) {
            e.preventDefault();
            alert('Пожалуйста, заполните все обязательные поля');
            return;
        }
        
        if (new Date(checkOut) <= new Date(checkIn)) {
            e.preventDefault();
            alert('Дата выезда должна быть позже даты заезда');
            return;
        }
    });
    
    // Функции
    function updateRoomInfo() {
        const roomId = roomSelect.value;
        
        if (roomId && roomData[roomId]) {
            const room = roomData[roomId];
            
            // Показываем информацию о номере
            roomInfo.style.display = 'block';
            document.getElementById('room-description').textContent = room.description;
            document.getElementById('room-max-guests').textContent = room.maxGuests;
            document.getElementById('room-bed-type').textContent = room.bedType;
            document.getElementById('room-size').textContent = room.roomSize || 'Не указан';
            document.getElementById('room-amenities').textContent = room.amenities || 'Не указаны';
            
            // Обновляем расчет стоимости
            updateBookingSummary();
            
            // Проверяем количество гостей
            checkGuestsCount();
        } else {
            roomInfo.style.display = 'none';
            bookingSummary.style.display = 'none';
        }
    }
    
    function updateBookingSummary() {
        const roomId = roomSelect.value;
        const checkIn = checkInInput.value;
        const checkOut = checkOutInput.value;
        
        if (!roomId || !checkIn || !checkOut) {
            bookingSummary.style.display = 'none';
            return;
        }
        
        const checkInDate = new Date(checkIn);
        const checkOutDate = new Date(checkOut);
        
        if (checkOutDate <= checkInDate) {
            bookingSummary.style.display = 'none';
            return;
        }
        
        const nights = Math.floor((checkOutDate - checkInDate) / (1000 * 60 * 60 * 24));
        const room = roomData[roomId];
        const totalPrice = nights * room.price;
        
        document.getElementById('nights-count').textContent = nights;
        document.getElementById('price-per-night').textContent = 
            formatPrice(room.price) + ' ₽';
        document.getElementById('total-price').textContent = 
            formatPrice(totalPrice) + ' ₽';
        
        bookingSummary.style.display = 'block';
    }
    
    function checkGuestsCount() {
        const roomId = roomSelect.value;
        if (!roomId) return;
        
        const room = roomData[roomId];
        const adults = parseInt(adultsInput.value) || 0;
        const children = parseInt(childrenInput.value) || 0;
        const totalGuests = adults + children;
        
        if (totalGuests > room.maxGuests) {
            guestWarning.style.display = 'block';
        } else {
            guestWarning.style.display = 'none';
        }
    }
    
    function formatPrice(price) {
        return Math.round(price).toString().replace(/\B(?=(\d{3})+(?!\d))/g, ' ');
    }
    
    // Установка минимальной даты для check_out при изменении check_in
    checkInInput.addEventListener('change', function() {
        if (this.value) {
            const minDate = new Date(this.value);
            minDate.setDate(minDate.getDate() + 1);
            checkOutInput.min = minDate.toISOString().split('T')[0];
            
            if (checkOutInput.value && new Date(checkOutInput.value) <= new Date(this.value)) {
                checkOutInput.value = '';
                updateBookingSummary();
            }
        }
    });
});
</script>