<?php

/* @var $this yii\web\View */
/* @var $model app\models\User */

use yii\helpers\Html;
use yii\helpers\Url;

$this->title = 'Регистрация';
?>
<div class="register-page">
    <h1 class="page-title"><?= Html::encode($this->title) ?></h1>

    <div class="register-form-container">
        <div class="form-column">
            <form id="register-form" class="auth-form" method="post" action="<?= Url::to(['/site/register']) ?>">
                <input type="hidden" name="<?= Yii::$app->request->csrfParam ?>" value="<?= Yii::$app->request->csrfToken ?>">
                
                <div class="form-group">
                    <label for="user-username" class="form-label">Логин *</label>
                    <input type="text" id="user-username" class="form-input" name="User[username]" 
                           value="<?= Html::encode($model->username) ?>" required
                           pattern="[a-zA-Z0-9]+" title="Только латинские буквы и цифры">
                    <div class="error-text"><?= Html::error($model, 'username') ?></div>
                </div>
                
                <div class="form-group">
                    <label for="user-email" class="form-label">Email *</label>
                    <input type="email" id="user-email" class="form-input" name="User[email]" 
                           value="<?= Html::encode($model->email) ?>" required>
                    <div class="error-text"><?= Html::error($model, 'email') ?></div>
                </div>
                
                <div class="form-group">
                    <label for="user-full_name" class="form-label">ФИО *</label>
                    <input type="text" id="user-full_name" class="form-input" name="User[full_name]" 
                           value="<?= Html::encode($model->full_name) ?>" required
                           pattern="[а-яА-ЯёЁ\s\-]+" title="Только русские буквы, пробелы и дефисы">
                    <div class="error-text"><?= Html::error($model, 'full_name') ?></div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Паспортные данные *</label>
                    <div class="passport-row">
                        <div class="passport-field">
                            <input type="text" id="user-passport_series" class="form-input" 
                                   name="User[passport_series]" placeholder="Серия"
                                   value="<?= Html::encode($model->passport_series) ?>" required
                                   maxlength="4" pattern="\d{4}" title="4 цифры">
                            <div class="error-text"><?= Html::error($model, 'passport_series') ?></div>
                        </div>
                        <div class="passport-field">
                            <input type="text" id="user-passport_number" class="form-input" 
                                   name="User[passport_number]" placeholder="Номер"
                                   value="<?= Html::encode($model->passport_number) ?>" required
                                   maxlength="6" pattern="\d{6}" title="6 цифр">
                            <div class="error-text"><?= Html::error($model, 'passport_number') ?></div>
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="user-address" class="form-label">Адрес проживания *</label>
                    <textarea id="user-address" class="form-input" name="User[address]" 
                              rows="3" required><?= Html::encode($model->address) ?></textarea>
                    <div class="error-text"><?= Html::error($model, 'address') ?></div>
                </div>
                
                <div class="form-group">
                    <label for="user-phone" class="form-label">Телефон *</label>
                    <input type="tel" id="user-phone" class="form-input" name="User[phone]" 
                           value="<?= Html::encode($model->phone) ?>" required
                           pattern="\+7\(\d{3}\)-\d{3}-\d{2}-\d{2}" 
                           placeholder="+7(999)-999-99-99">
                    <div class="form-hint">Формат: +7(XXX)-XXX-XX-XX</div>
                    <div class="error-text"><?= Html::error($model, 'phone') ?></div>
                </div>
                
                <div class="form-group">
                    <label for="user-password" class="form-label">Пароль *</label>
                    <input type="password" id="user-password" class="form-input" 
                           name="User[password]" required minlength="7">
                    <div class="form-hint">Минимум 7 символов</div>
                    <div class="error-text"><?= Html::error($model, 'password') ?></div>
                </div>
                
                <div class="form-group">
                    <label for="user-password_repeat" class="form-label">Повторите пароль *</label>
                    <input type="password" id="user-password_repeat" class="form-input" 
                           name="User[password_repeat]" required>
                    <div class="error-text"><?= Html::error($model, 'password_repeat') ?></div>
                </div>

                <div class="form-group">
                    <button type="submit" class="submit-btn">Зарегистрироваться</button>
                </div>
            </form>
            
            <div class="auth-links">
                <p>Уже есть аккаунт? <a href="<?= Url::to(['/site/login']) ?>">Войдите</a></p>
            </div>
        </div>
        
        <div class="info-column">
            <div class="info-card">
                <h3 class="info-title">Требования к данным</h3>
                <ul class="requirements-list">
                    <li><strong>Логин:</strong> только латинские буквы и цифры</li>
                    <li><strong>Email:</strong> должен быть уникальным и действительным</li>
                    <li><strong>ФИО:</strong> только русские буквы, пробелы и дефисы</li>
                    <li><strong>Серия паспорта:</strong> 4 цифры</li>
                    <li><strong>Номер паспорта:</strong> 6 цифр</li>
                    <li><strong>Телефон:</strong> формат +7(XXX)-XXX-XX-XX</li>
                    <li><strong>Пароль:</strong> минимум 7 символов</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<style>
.register-page {
    max-width: 1000px;
    margin: 0 auto;
    animation: fadeIn 0.5s ease;
}

.page-title {
    text-align: center;
    margin-bottom: 2rem;
    color: #2c3e50;
    font-size: 2rem;
}

.register-form-container {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
}

@media (max-width: 768px) {
    .register-form-container {
        grid-template-columns: 1fr;
    }
}

.auth-form {
    background: white;
    padding: 2rem;
    border-radius: 8px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #34495e;
}

.form-input {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 1rem;
    transition: all 0.3s;
}

.form-input:focus {
    outline: none;
    border-color: #3498db;
    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.2);
}

.form-input.error {
    border-color: #e74c3c;
}

.passport-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.passport-field {
    display: flex;
    flex-direction: column;
}

textarea.form-input {
    resize: vertical;
    min-height: 80px;
}

.form-hint {
    font-size: 0.85rem;
    color: #7f8c8d;
    margin-top: 0.25rem;
}

.error-text {
    color: #e74c3c;
    font-size: 0.85rem;
    margin-top: 0.25rem;
    min-height: 1.2rem;
}

.submit-btn {
    width: 100%;
    padding: 1rem;
    background: linear-gradient(135deg, #3498db, #2980b9);
    color: white;
    border: none;
    border-radius: 4px;
    font-size: 1.1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
}

.submit-btn:hover {
    background: linear-gradient(135deg, #2980b9, #1c6ea4);
    transform: translateY(-2px);
    box-shadow: 0 4px 10px rgba(0,0,0,0.2);
}

.submit-btn:active {
    transform: translateY(0);
}

.auth-links {
    margin-top: 1.5rem;
    text-align: center;
}

.auth-links a {
    color: #3498db;
    text-decoration: none;
}

.auth-links a:hover {
    text-decoration: underline;
}

.info-card {
    background: white;
    padding: 2rem;
    border-radius: 8px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.info-title {
    color: #2c3e50;
    margin-bottom: 1rem;
    font-size: 1.3rem;
}

.requirements-list {
    list-style: none;
}

.requirements-list li {
    margin-bottom: 0.75rem;
    padding-left: 1.5rem;
    position: relative;
}

.requirements-list li:before {
    content: "✓";
    position: absolute;
    left: 0;
    color: #27ae60;
    font-weight: bold;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('register-form');
    const phoneInput = document.getElementById('user-phone');
    const seriesInput = document.getElementById('user-passport_series');
    const numberInput = document.getElementById('user-passport_number');
    const passwordInput = document.getElementById('user-password');
    const passwordRepeatInput = document.getElementById('user-password_repeat');
    
    // Маска для телефона
    phoneInput.addEventListener('input', function(e) {
        let value = e.target.value.replace(/\D/g, '');
        if (value.length > 0) {
            value = '+7(' + value.substring(1, 4) + ')-' + 
                    value.substring(4, 7) + '-' + 
                    value.substring(7, 9) + '-' + 
                    value.substring(9, 11);
        }
        e.target.value = value.substring(0, 17);
    });
    
    // Ограничение для серии паспорта
    seriesInput.addEventListener('input', function(e) {
        e.target.value = e.target.value.replace(/\D/g, '').substring(0, 4);
    });
    
    // Ограничение для номера паспорта
    numberInput.addEventListener('input', function(e) {
        e.target.value = e.target.value.replace(/\D/g, '').substring(0, 6);
    });
    
    // Валидация совпадения паролей
    passwordRepeatInput.addEventListener('input', function() {
        if (passwordInput.value !== passwordRepeatInput.value) {
            passwordRepeatInput.setCustomValidity('Пароли не совпадают');
        } else {
            passwordRepeatInput.setCustomValidity('');
        }
    });
    
    // AJAX валидация формы
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Сброс предыдущих ошибок
        document.querySelectorAll('.error-text').forEach(el => el.textContent = '');
        document.querySelectorAll('.form-input').forEach(el => el.classList.remove('error'));
        
        const formData = new FormData(form);
        
        fetch('<?= Url::to(['/site/register']) ?>', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (Object.keys(data).length === 0) {
                // Нет ошибок, отправляем форму
                form.submit();
            } else {
                // Показываем ошибки
                Object.keys(data).forEach(field => {
                    const errorElement = document.querySelector(`[data-error="${field}"]`) || 
                                         document.querySelector(`#user-${field}`)?.nextElementSibling;
                    if (errorElement) {
                        errorElement.textContent = data[field][0];
                        errorElement.previousElementSibling?.classList.add('error');
                    }
                });
                
                // Прокрутка к первой ошибке
                const firstError = document.querySelector('.error');
                if (firstError) {
                    firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
                }
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Произошла ошибка при отправке формы');
        });
    });
    
    // Добавляем data-атрибуты для ошибок
    document.querySelectorAll('.error-text').forEach(el => {
        const inputId = el.previousElementSibling?.id;
        if (inputId) {
            const fieldName = inputId.replace('user-', '');
            el.setAttribute('data-error', fieldName);
        }
    });
});
</script>